#!/usr/bin/env python3

# imports
import argparse
import json
import logging
import sys
import typing

# bsfs imports
import bsfs

# exports
__all__: typing.Sequence[str] = (
    'main',
    )


## code ##

logger = logging.getLogger(__name__)

def main(argv):
    """Migrate a storage structure to a modified schema."""
    parser = argparse.ArgumentParser(description=main.__doc__, prog='migrate')
    parser.add_argument('--remove', action='store_true', default=False,
        help='Remove classes that are not specified in the provided schema.')
    parser.add_argument('config', type=str, default=None,
        help='Path to the storage config file.')
    parser.add_argument('schema', nargs=argparse.REMAINDER,
        help='Paths to schema files. Reads from standard input if no file is supplied.')
    args = parser.parse_args(argv)

    # load storage config
    with open(args.config, mode='rt', encoding='UTF-8') as ifile:
        config = json.load(ifile)
    # open bsfs storage
    graph = bsfs.Open(config)

    # initialize schema
    schema = bsfs.schema.Schema()
    if len(args.schema) == 0:
        # assemble schema from standard input
        schema = schema + bsfs.schema.from_string(sys.stdin.read())
    else:
        # assemble schema from input files
        for pth in args.schema:
            with open(pth, mode='rt', encoding='UTF-8') as ifile:
                schema = schema + bsfs.schema.from_string(ifile.read())

    # migrate schema
    graph.migrate(schema, not args.remove)

    # return the migrated storage
    return graph


## main ##

if __name__ == '__main__':
    main(sys.argv[1:])

## EOF ##
