
# imports
import abc
import typing

# bsfs imports
from bsfs import schema
from bsfs.query import ast
from bsfs.triple_store import TripleStoreBase
from bsfs.utils import URI, typename

# exports
__all__: typing.Sequence[str] = (
    'AccessControlBase',
    )


## code ##

class AccessControlBase(abc.ABC):
    """Defines the interface for access control policies.

    An access control policy governs which actions a user may take to query
    or to manipulate a graph.

    """

    # The triple store backend.
    _backend: TripleStoreBase

    # The current user.
    _user: URI

    def __init__(
            self,
            backend: TripleStoreBase,
            user: URI,
            ):
        self._backend = backend
        self._user = URI(user)

    def __str__(self) -> str:
        return f'{typename(self)}({self._user})'

    def __repr__(self) -> str:
        return f'{typename(self)}({self._user})'

    def __eq__(self, other: typing.Any) -> bool:
        return isinstance(other, type(self)) \
           and self._backend == other._backend \
           and self._user == other._user

    def __hash__(self) -> int:
        return hash((type(self), self._backend, self._user))

    @abc.abstractmethod
    def is_protected_predicate(self, pred: schema.Predicate) -> bool:
        """Return True if a predicate cannot be modified manually."""

    @abc.abstractmethod
    def create(self, node_type: schema.Node, guids: typing.Iterable[URI]):
        """Perform post-creation operations on nodes, e.g. ownership information."""

    @abc.abstractmethod
    def link_from_node(self, node_type: schema.Node, guids: typing.Iterable[URI]) -> typing.Iterable[URI]:
        """Return nodes for which outbound links can be written."""

    @abc.abstractmethod
    def link_to_node(self, node_type: schema.Node, guids: typing.Iterable[URI]) -> typing.Iterable[URI]:
        """Return nodes for which inbound links can be written."""

    @abc.abstractmethod
    def write_literal(self, node_type: schema.Node, guids: typing.Iterable[URI]) -> typing.Iterable[URI]:
        """Return nodes to which literals can be attached."""

    @abc.abstractmethod
    def createable(self, node_type: schema.Node, guids: typing.Iterable[URI]) -> typing.Iterable[URI]:
        """Return nodes that are allowed to be created."""

    @abc.abstractmethod
    def filter_read(
            self,
            node_type: schema.Node,
            query: typing.Optional[ast.filter.FilterExpression],
            ) -> typing.Optional[ast.filter.FilterExpression]:
        """Re-write a filter *query* to get (i.e., read) *node_type* nodes."""

    @abc.abstractmethod
    def fetch_read(self, node_type: schema.Node, query: ast.fetch.FetchExpression) -> ast.fetch.FetchExpression:
        """Re-write a fetch *query* to get (i.e, read) values for *node_type* nodes."""

## EOF ##
