
# imports
import io
import typing

# external imports
import PIL.Image

# bsie imports
from bsie.utils import bsfs, node, ns

# inner-module imports
from . import base

# exports
__all__: typing.Sequence[str] = (
    'Preview',
    )


## code ##

class Preview(base.Extractor):
    """Extract previews."""

    CONTENT_READER = 'bsie.reader.preview.Preview'

    def __init__(self, max_sides: typing.Iterable[int]):
        super().__init__(bsfs.schema.from_string(base.SCHEMA_PREAMBLE + '''



            bsn:Preview rdfs:subClassOf bsfs:Node .
            bsl:BinaryBlob rdfs:subClassOf bsfs:Literal .
            <https://schema.bsfs.io/ie/Literal/BinaryBlob/JPEG> rdfs:subClassOf bsl:BinaryBlob .

            bse:preview rdfs:subClassOf bsfs:Predicate ;
                rdfs:domain bsn:Entity ;
                rdfs:range bsn:Preview ;
                bsfs:unique "false"^^xsd:boolean .

            bsp:width rdfs:subClassOf bsfs:Predicate ;
                rdfs:domain bsn:Preview ;
                rdfs:range xsd:integer ;
                bsfs:unique "true"^^xsd:boolean .

            bsp:height rdfs:subClassOf bsfs:Predicate ;
                rdfs:domain bsn:Preview ;
                rdfs:range xsd:integer ;
                bsfs:unique "true"^^xsd:boolean .

            bsp:asset rdfs:subClassOf bsfs:Predicate ;
                rdfs:domain bsn:Preview ;
                rdfs:range <https://schema.bsfs.io/ie/Literal/BinaryBlob/JPEG> ;
                bsfs:unique "true"^^xsd:boolean .

            '''))
        # initialize extra args
        self.max_sides = set(max_sides)

    def __eq__(self, other: typing.Any) -> bool:
        return super().__eq__(other) \
           and self.max_sides == other.max_sides

    def __hash__(self) -> int:
        return hash((super().__hash__(), tuple(sorted(self.max_sides))))

    def extract(
            self,
            subject: node.Node,
            content: typing.Callable[[int], PIL.Image.Image],
            principals: typing.Iterable[bsfs.schema.Predicate],
            ) -> typing.Iterator[typing.Tuple[node.Node, bsfs.schema.Predicate, typing.Any]]:
        # check principals
        if self.schema.predicate(ns.bse.preview) not in principals:
            return

        for max_side in self.max_sides:
            # get the preview in the right resolution
            img = content(max_side)
            # convert the preview to jpeg
            buffer = io.BytesIO()
            img.save(buffer, format='jpeg')
            # create a preview node
            preview = node.Node(ns.bsn.Preview,
                ucid=bsfs.uuid.UCID.from_bytes(buffer.getvalue()),
                size=max_side,
                source=subject,
                )
            # yield triples
            yield subject, self.schema.predicate(ns.bse.preview), preview
            yield preview, self.schema.predicate(ns.bsp.width), img.width
            yield preview, self.schema.predicate(ns.bsp.height), img.height
            yield preview, self.schema.predicate(ns.bsp.asset), buffer.getvalue()

## EOF ##
