"""Tagit standalone user interfaces.

Tagit contains several submodules:
* apps: Launchable standalone applications.
* windows: Compositions of various UI functionality.
* widgets: Individual UI components and core functions.
* dialogues: Pop-up dialogues.
* actions: Single actions that can be performed on widgets.
* tiles: Display a digestible amount of information.

Additional core functionality is provided by some generic
modules (utils, parsing, config, etc.).

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
import collections
import os
import typing

import kivy.config
kivy.config.Config.set('input', 'mouse', 'mouse,disable_multitouch')

# kivy imports
from kivy.resources import resource_add_path
import kivy

# constants
T_VERSION_INFO = collections.namedtuple('T_VERSION_INFO', ('major', 'minor', 'micro'))
version_info = T_VERSION_INFO(0, 0, 1)

# exports
__all__: typing.Sequence[str] = []


## code ##

# check kivy version
kivy.require('1.9.1')

# add resources
resource_add_path(os.path.join(os.path.dirname(__file__), 'assets', 'icons', 'kivy'))
resource_add_path(os.path.join(os.path.dirname(__file__), 'assets', 'fonts', 'kivy'))
resource_add_path(os.path.join(os.path.dirname(__file__), 'assets', 'themes'))
resource_add_path(os.path.join(os.path.dirname(__file__), 'assets'))

# load font
from kivy.core.text import LabelBase
LabelBase.register(name='Unifont', fn_regular='Unifont.ttf')

# logging
# the default logger is quite verbose. Should be restricted by the app.
import logging
from . import logger
loghandler = logger.logger_config(
    handler=logging.StreamHandler(),
    colors=logger.ColorsTerminal,
    config=dict(
        level='DEBUG',
        fmt='[{levelname}] [{module:<12}] {title}{message}',
        title='[{title}] ',
        filter=['tagit'],
        )
    )

termlogger = logging.getLogger(__name__)
termlogger.addHandler(loghandler)
termlogger.setLevel(logging.DEBUG)

# console logging fix:
# kivy adds an extra whitespace in front of tagit log entries.
# Adding a carriage return in front of the log fixes this bug.
# This is only needed for the console log handler, not others.
# Note that this mechanism is repeated in apps/gui to
# achieve the same for user-defined log handlers.
loghandler.formatter.prefix = '\r'

## EOF ##
