"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
import os

# kivy imports
from kivy.lang import Builder
import kivy.properties as kp

# tagit imports
from tagit import config, dialogues
from tagit.utils import errors, ns, Frame
from tagit.utils.bsfs import ast
from tagit.widgets import Binding
from tagit.widgets.filter import FilterAwareMixin

# inner-module imports
from .action import Action

# exports
__all__ = []


## code ##

# load kv
Builder.load_file(os.path.join(os.path.dirname(__file__), 'filter.kv'))

# classes

class SearchByAddressOnce(Action):
    """Open the filters in address mode for a single edit"""
    text = kp.StringProperty('Inline edit')

    def ktrigger(self, evt):
        return Binding.check(evt, self.cfg('bindings', 'filter', 'edit_once'))

    def apply(self):
        self.root.filter.show_address_once()


class SetToken(Action):
    """Set all filters from a text query."""
    text = kp.StringProperty('Set tokens')

    def apply(self, text):
        with self.root.filter as filter:
            try:
                # parse filter into tokens
                tokens = list(self.root.session.filter_from_string(text))

                # grab current frame
                filter.f_head.append(self.root.browser.frame)

                # keep frames for tokens that didn't change
                # create a new frame for changed (new or modified) tokens
                frames = [filter.f_head[0]]
                for tok in tokens:
                    if tok in filter.t_head: # t_head frames have precedence
                        frame = filter.f_head[filter.t_head.index(tok) + 1]
                    elif tok in filter.t_tail:
                        frame = filter.f_tail[filter.t_tail.index(tok)]
                    else:
                        frame = Frame()
                    frames.append(frame)
                # use the last frame as current one
                self.root.browser.frame = frames.pop()

                # set tokens
                filter.t_head = tokens
                filter.t_tail = []
                # set frames
                filter.f_head = frames
                filter.f_tail = []

            except ParserError as e:
                dialogues.Error(text=f'syntax error: {e}').open()


class AddToken(Action):
    """Show a dialogue for adding a filter."""
    text = kp.StringProperty('Add filter')

    def ktrigger(self, evt):
        return Binding.check(evt, self.cfg('bindings', 'filter', 'add_token'))

    def apply(self, token=None):
        if token is None:
            sugg = self.root.session.storage.all(ns.bsn.Tag).label(node=False)
            dlg = dialogues.TokenEdit(suggestions=sugg)
            dlg.bind(on_ok=lambda wx: self.add_from_string(wx.text))
            dlg.open()
        elif isinstance(token, str):
            self.add_from_string(token)
        elif isinstance(token, ast.filter.FilterExpression):
            self.add_token([token])

    def add_from_string(self, text):
        try:
            self.add_token(self.root.session.filter_from_string(text))
        except errors.ParserError as e:
            dialogues.Error(text=f'syntax error: {e}').open()

    def add_token(self, tokens):
        with self.root.filter as filter:
            tokens = [tok for tok in tokens if tok not in filter.t_head]
            for tok in tokens:
                # add token and frame
                filter.t_head.append(tok)
                filter.f_head.append(self.root.browser.frame)
            if len(tokens):
                # clear tails
                filter.t_tail = []
                filter.f_tail = []
                # issue new frame
                self.root.browser.frame = Frame()


class EditToken(Action):
    """Show a dialogue for editing a filter."""
    text = kp.StringProperty('Edit token')

    def apply(self, token):
        sugg = self.root.session.storage.all(ns.bsn.Tag).label(node=False)
        text = self.root.session.filter_to_string(token)
        dlg = dialogues.TokenEdit(text=text, suggestions=sugg)
        dlg.bind(on_ok=lambda obj: self.on_ok(token, obj))
        dlg.open()

    def on_ok(self, token, obj):
        with self.root.filter as filter:
            try:
                tokens_from_text = self.root.session.filter_from_string(obj.text)
            except errors.ParserError as e:
                dialogues.Error(text=f'Invalid token: {e}').open()
                return

            # TODO: Check if this can be simplified
            keep = False
            for tok in tokens_from_text:
                if tok == token: # don't remove if the token hasn't changed
                    keep = True

                if token in filter.t_head and tok not in filter.t_head:
                    # insert after token into t_head
                    idx = filter.t_head.index(token)
                    frame = filter.f_head[idx]
                    filter.t_head.insert(idx + 1, tok)
                    filter.f_head.insert(idx + 1, frame.copy())
                    # remove from t_tail
                    if tok in filter.t_tail:
                        idx = filter.t_tail.index(tok)
                        filter.f_tail.pop(idx)
                        filter.t_tail.pop(idx)
                elif token in filter.t_tail and tok not in filter.t_tail:
                    # insert after token into t_tail
                    idx = filter.t_tail.index(token)
                    frame = filter.f_tail[idx]
                    filter.t_tail.insert(idx + 1, tok)
                    filter.f_tail.insert(idx + 1, frame.copy())
                    # remove from t_head
                    if tok in filter.t_head:
                        idx = filter.t_head.index(tok)
                        filter.t_head.pop(idx)
                        filter.f_head.pop(idx)

            # remove original token
            if not keep and token in filter.t_head:
                idx = filter.t_head.index(token)
                filter.t_head.pop(idx)
                filter.f_head.pop(idx)
            if not keep and token in filter.t_tail:
                idx = filter.t_tail.index(token)
                filter.t_tail.pop(idx)
                filter.f_tail.pop(idx)


class RemoveToken(Action):
    """Remove a filter."""
    text = kp.StringProperty('Remove token')

    def apply(self, token):
        with self.root.filter as filter:
            if token in filter.t_head:
                idx = filter.t_head.index(token)
                # remove frame
                if idx < len(filter.t_head) - 1:
                    filter.f_head.pop(idx + 1)
                    self.root.browser.frame = Frame()
                else:
                    self.root.browser.frame = filter.f_head.pop()
                # remove token
                filter.t_head.remove(token)

            if token in filter.f_tail:
                filter.f_tail.pop(filter.t_tail.index(token))
                filter.t_tail.remove(token)


class GoBack(Action):
    """Remove the rightmost filter from the search."""
    text = kp.StringProperty('Previous search')

    def ktrigger(self, evt):
        return Binding.check(evt, self.cfg('bindings', 'filter', 'go_back'))

    def apply(self, n_steps=1):
        with self.root.filter as filter:
            for _ in range(n_steps):
                if len(filter.t_head) > 0:
                    # move tokens
                    filter.t_tail.insert(0, filter.t_head.pop(-1))
                    # move frames
                    filter.f_tail.insert(0, self.root.browser.frame)
                    self.root.browser.frame = filter.f_head.pop(-1)


class GoForth(Action):
    """Add the rightmost filter to the search"""
    text = kp.StringProperty('Next search')

    def ktrigger(self, evt):
        return Binding.check(evt, self.cfg('bindings', 'filter', 'go_forth'))

    def apply(self, n_steps=1):
        with self.root.filter as filter:
            for _ in range(n_steps):
                if len(filter.t_tail) > 0:
                    # move tokens
                    filter.t_head.append(filter.t_tail.pop(0))
                    # move frames
                    filter.f_head.append(self.root.browser.frame)
                    self.root.browser.frame = filter.f_tail.pop(0)


class JumpToToken(Action):
    """Jump to a filter token."""
    text = kp.StringProperty('Jump to token')

    def apply(self, token):
        filter = self.root.filter
        if token == filter.t_head[-1]:
            pass
        elif token in filter.t_head: # go
            self.root.trigger('GoBack', len(filter.t_head) - filter.t_head.index(token) - 1)
        elif token in filter.t_tail:
            self.root.trigger('GoForth', filter.t_tail.index(token) + 1)


class SearchmodeSwitch(Action, FilterAwareMixin):
    """Switch between shingle and address search bar display."""
    text = kp.StringProperty('Toggle searchbar mode')

    def on_root(self, wx, root):
        Action.on_root(self, wx, root)
        FilterAwareMixin.on_root(self, wx, root)

    def on_searchmode(self, filter, searchmode):
        if self._image is not None:
            if searchmode == filter.MODE_ADDRESS:
                self._image.source = self.source_address
            else:
                self._image.source = self.source_shingles

    def on_filter(self, wx, filter):
        # remove old binding
        if self.filter is not None:
            self.filter.unbind(searchmode=self.on_searchmode)
        # add new binding
        self.filter = filter
        if self.filter is not None:
            self.filter.bind(searchmode=self.on_searchmode)
            self.on_searchmode(self.filter, self.filter.searchmode)

    def __del__(self):
        if self.filter is not None:
            self.filter.unbind(searchmode=self.on_searchmode)
            self.filter = None

    def apply(self):
        with self.root.filter as filter:
            if filter.searchmode == filter.MODE_SHINGLES:
                filter.searchmode = filter.MODE_ADDRESS
            else:
                filter.searchmode = filter.MODE_SHINGLES


## config ##

# keybindings

config.declare(('bindings', 'filter', 'add_token'),
    config.Keybind(), Binding.simple('k', Binding.mCTRL, Binding.mREST), # Ctrl + k
    __name__, AddToken.text.defaultvalue, AddToken.__doc__)

config.declare(('bindings', 'filter', 'go_forth'),
    config.Keybind(), Binding.simple(Binding.RIGHT, Binding.mALT, Binding.mREST), # Alt + right
    __name__, GoForth.text.defaultvalue, GoForth.__doc__)

config.declare(('bindings', 'filter', 'edit_once'), config.Keybind(),
    Binding.multi(('l', Binding.mCTRL, Binding.mREST),
                  ('/', Binding.mREST, Binding.mALL)), # Ctrl + l, /
    __name__, SearchByAddressOnce.text.defaultvalue, SearchByAddressOnce.__doc__)

config.declare(('bindings', 'filter', 'go_back'), config.Keybind(),
    Binding.multi((Binding.BACKSPACE, Binding.mCTRL, Binding.mREST),
                  (Binding.LEFT, Binding.mALT, Binding.mREST)), # Ctrl + backspace, Alt + left
    __name__, GoBack.text.defaultvalue, GoBack.__doc__)

## EOF ##
