"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
import code
import logging
import os
import sys

# external imports
import webbrowser

# kivy imports
from kivy.core.clipboard import Clipboard
from kivy.lang import Builder
import kivy.properties as kp

# tagit imports
from tagit import config
from tagit.utils import fileopen
from tagit.widgets import Binding

# inner-module imports
from .action import Action

# constants
HELP_URL = 'https://www.bsfs.io/'

# exports
__all__ = []


## code ##

logger = logging.getLogger(__name__)

# load kv
Builder.load_file(os.path.join(os.path.dirname(__file__), 'misc.kv'))

# classes
class Menu(Action):
    """Open the menu."""
    text = kp.StringProperty('Menu')

    def ktrigger(self, evt):
        return Binding.check(evt, self.cfg('bindings', 'misc', 'menu'))

    def apply(self):
        x = self.pos[0] + self.width
        y = self.pos[1] + self.height
        self.root.context.show(x, y)


class ShellDrop(Action):
    """Open a terminal shell."""
    text = kp.StringProperty('Shell')

    def apply(self):
        from pprint import pprint as pp
        loc = globals()
        loc.update(locals())
        code.interact(banner='tagit shell', local=loc)
        if loc.get('abort', False):
            sys.exit(1)



class OpenExternal(Action):
    """Open the selected items in an external application."""
    text = kp.StringProperty('Open')

    def ktrigger(self, evt):
        # FIXME: Triggered on Shift + Click (Interferes with selection!)
        # Triggered on <Enter> when tags are edited.
        return Binding.check(evt, self.cfg('bindings', 'misc', 'open'))

    def apply(self):
        return # FIXME: mb/port
        with self.root.browser as browser:
            if browser.cursor is None:
                logger.error('No file selected')
            elif os.path.exists(browser.cursor.path):
                fileopen(browser.cursor.path)
            else:
                logger.error('File unavailable')


class ShowConsole(Action):
    """Open the log console."""
    text = kp.StringProperty('Console')

    def apply(self):
        self.root.status.console()


class ShowHelp(Action):
    """Show some help."""
    text = kp.StringProperty('Help')

    def ktrigger(self, evt):
        return Binding.check(evt, self.cfg('bindings', 'misc', 'help'))

    def apply(self):
        webbrowser.open(HELP_URL)


class ShowSettings(Action):
    """Open the settings menu."""
    text = kp.StringProperty('Settings')

    def ktrigger(self, evt):
        return Binding.check(evt, self.cfg('bindings', 'misc', 'settings'))

    def apply(self):
        from kivy.app import App
        App.get_running_app().open_settings()


class ClipboardCopy(Action):
    """Copy selected items into the clipboard."""
    text = kp.StringProperty('Copy to clipboard')

    def ktrigger(self, evt):
        return Binding.check(evt, self.cfg('bindings', 'clipboard', 'copy'))

    def apply(self):
        return # FIXME: mb/port
        browser = self.root.browser
        paths = [obj.path for obj in browser.selection]
        Clipboard.copy('\n'.join(paths))


class ClipboardPaste(Action):
    """Import items from the clipboard."""
    text = kp.StringProperty('Paste from clipboard')

    def ktrigger(self, evt):
        return Binding.check(evt, self.cfg('bindings', 'clipboard', 'paste'))

    def apply(self):
        return # FIXME: mb/port
        paths = Clipboard.paste()
        paths = paths.split('\n')
        self.root.trigger('ImportObjects', paths)


## config ##

# keybindings

config.declare(('bindings', 'misc', 'menu'),
    config.Keybind(), Binding.simple(Binding.CMD, None, Binding.mALL),
    __name__, Menu.text.defaultvalue, Menu.__doc__)

config.declare(('bindings', 'misc', 'open'),
    config.Keybind(), Binding.simple(Binding.ENTER, None, Binding.mALL),
    __name__, OpenExternal.text.defaultvalue, OpenExternal.__doc__)

config.declare(('bindings', 'misc', 'help'),
    config.Keybind(), Binding.simple('/', Binding.mSHIFT),
    __name__, ShowHelp.text.defaultvalue, ShowHelp.__doc__)

config.declare(('bindings', 'misc', 'settings'),
    config.Keybind(), Binding.simple(Binding.F1), # also the kivy default
    __name__, ShowSettings.text.defaultvalue, ShowSettings.__doc__)

config.declare(('bindings', 'clipboard', 'copy'),
    config.Keybind(), Binding.simple('c', Binding.mCTRL),
    __name__, ClipboardCopy.text.defaultvalue, ClipboardCopy.__doc__)

config.declare(('bindings', 'clipboard', 'paste'),
    config.Keybind(), Binding.simple('v', Binding.mCTRL),
    __name__, ClipboardPaste.text.defaultvalue, ClipboardPaste.__doc__)

## EOF ##
