"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
import typing

# tagit imports
from tagit.utils import errors, is_list

# exports
__all__: typing.Sequence[str] = (
    'ConfigTypeError',
    # types
    'Any',
    'Bool',
    'Dict',
    'Enum',
    'Float',
    'Int',
    'Keybind',
    'List',
    'Numeric',
    'Path',
    'String',
    'Unsigned',
    )

# TODO: Bounded int or range? (specify lo/hi bounds)
# TODO: File vs. Dir; existence condition?

## code ##

# base class

class ConfigTypeError(TypeError):
    """Raised if a type inconsistency is detected."""
    pass

class ConfigType(object):
    """A config type defines a constraint over admissible values in order to
    perform a basic verification of user-entered config values.
    """

    # example values
    example = ''

    # type description
    description = ''

    def __str__(self):
        return f'{type(self).__name__}'

    def __repr__(self):
        return f'{type(self).__name__}()'

    def __eq__(self, other):
        return isinstance(other, type(self))

    def __hash__(self):
        return hash(type(self))

    def check(self, value):
        """Return True if the *value* matches the type."""
        try:
            self.backtrack(value, '')
            return True
        except ConfigTypeError:
            return False

    def backtrack(self, value, key):
        """Check *value* for errors.
        Raises a ConfigTypeError with a detailed message if an inconsistency is detected.
        """
        errors.abstract()

# generic types

class Any(ConfigType):
    example = '1, "a", [1,2,"a"]'
    description = 'Any type'

    def backtrack(self, value, key):
        # accepts anything
        pass


class Bool(ConfigType):
    example = 'True, False'
    description = 'Boolean'

    def backtrack(self, value, key):
        if not isinstance(value, bool):
            raise ConfigTypeError(f'found {value} in {key}, expected a boolean')


class Keybind(ConfigType):
    example = '[("a", ["ctrl"], [])]'
    description = 'A list of (key, required modifiers, excluded modifiers)-triples'

    def backtrack(self, value, key):
        if not is_list(value):
            raise ConfigTypeError(f'found {type(value)} in {key}, expected a list of bindings')

        modifiers = {'shift', 'alt', 'ctrl', 'cmd', 'altgr', 'rest', 'all'}
        for idx, itm in enumerate(value):
            if not is_list(itm) or len(itm) != 3:
                raise ConfigTypeError(f'found {itm} in {key}[{idx}], expected a list of three')

            char, inc, exc = itm
            if not isinstance(char, str) and \
               not isinstance(char, int) and \
               not isinstance(char, float):
                raise ConfigTypeError(
                        f'found {char} in {key}[{idx}], expected a character or number')
            if not is_list(inc) or not set(inc).issubset(modifiers):
                mods = ','.join(modifiers)
                raise ConfigTypeError(f'found {inc} in {key}[{idx}], expected some of ({mods})')
            if not is_list(exc) or not set(exc).issubset(modifiers):
                mods = ','.join(modifiers)
                raise ConfigTypeError(f'found {exc} in {key}[{idx}], expected some of ({mods})')


# numeric types

class Numeric(ConfigType):
    pass


class Int(Numeric):
    example = '-8, -1, 0, 1, 3'
    description = 'Integer number'

    def backtrack(self, value, key):
        if not isinstance(value, int):
            raise ConfigTypeError(f'found {value} in {key}, expected an integer')


class Unsigned(Int):
    example = '0, 1, 13, 32'
    description = 'Non-negative integer number, including zero'

    def __str__(self):
        return 'Unsigned int'

    def backtrack(self, value, key):
        if not isinstance(value, int) or value < 0:
            raise ConfigTypeError(f'found {value} in {key}, expeced an integer of at least zero')


class Float(Numeric):
    example = '1.2, 3.4, 5, 6'
    description = 'Integer or Decimal number'

    def backtrack(self, value, key):
        if not isinstance(value, float) and not isinstance(value, int):
            raise ConfigTypeError(f'found {value} in {key}, expected a number')


# string types

class String(ConfigType):
    example = '"hello world", "", "foobar"'
    description = 'String'

    def backtrack(self, value, key):
        if not isinstance(value, str):
            raise ConfigTypeError(f'found {value} in {key}, expected a string')


class Path(String):
    example = '"/tmp", "Pictures/trip", "~/.tagitrc"'
    description = 'String, compliant with file system paths'


# compound types

class Enum(ConfigType):
    description = 'One out of a predefined set of values'

    @property
    def example(self):
        return ', '.join(str(o) for o in list(self.options)[:3])

    def __init__(self, *options):
        self.options = set(options[0] if len(options) == 1 and is_list(options[0]) else options)

    def __eq__(self, other):
        return super(Enum, self).__eq__(other) and \
               self.options == other.options

    def __hash__(self):
        return hash((super(Enum, self).__hash__(), tuple(self.options)))

    def __str__(self):
        options = ', '.join(str(itm) for itm in self.options)
        return f'One out of ({options})'

    def __repr__(self):
        return f'{type(self).__name__}([{self.options}])'

    def backtrack(self, value, key):
        try:
            if value not in self.options:
                raise Exception()
        except Exception:
            options = ','.join(str(itm) for itm in self.options)
            raise ConfigTypeError(f'found {value} in {key}, expected one out of ({options})')


class List(ConfigType):
    description = 'List of values'

    @property
    def example(self):
        return f'[{self.item_type.example}]'

    def __init__(self, item_type):
        self.item_type = item_type

    def __eq__(self, other):
        return super(List, self).__eq__(other) and \
               self.item_type == other.item_type

    def __hash__(self):
        return hash((super(List, self).__hash__(), hash(self.item_type)))

    def __str__(self):
        return f'List of {str(self.item_type)}'

    def __repr__(self):
        return f'{type(self).__name__}({self.item_type})'

    def backtrack(self, value, key):
        if not isinstance(value, list) and not isinstance(value, tuple):
            raise ConfigTypeError(f'found {type(value)} in {key}, expected list')
        for item in value:
            self.item_type.backtrack(item, key)


class Dict(ConfigType):
    example = '{"hello": "world"}; {"hello": 3}; {"hello": [1, 2, 3]}'
    description = 'Map of keys/values'

    def __init__(self, key_type, value_type):
        self.key_type = key_type
        self.value_type = value_type

    def __eq__(self, other):
        return super(Dict, self).__eq__(other) and \
               self.key_type == other.key_type and \
               self.value_type == other.value_type

    def __hash__(self):
        return hash((super(Dict, self).__hash__(), hash(self.key_type), hash(self.value_type)))

    def __str__(self):
        return f'Dict from {self.key_type} to {self.value_type}'


    def __repr__(self):
        return f'{type(self).__name__}({self.key_type}, {self.value_type})'

    def backtrack(self, value, key):
        if not isinstance(value, dict):
            raise ConfigTypeError(f'found {type(value)} in {key}, expected a dict')
        for subkey, subval in value.items():
            self.key_type.backtrack(subkey, str(key) + '.' + str(subkey))
            self.value_type.backtrack(subval, str(key) + '.' + str(subkey))

## EOF ##
