"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
import re

# exports
__all__ = ('Colors', 'ColorsTerminal', 'ColorsMarkup')


## code ##

class Colors(object):
    """

    Status:
    * ok
    * warn
    * error

    Messages:
    * title
    * info

    """

    ## status ##

    @classmethod
    def ok(obj, text):
        return obj.BOLD + obj.OK + text + obj.ENDC + obj.ENDB

    @classmethod
    def warn(obj, text):
        return obj.BOLD + obj.WARN + text + obj.ENDC + obj.ENDB

    @classmethod
    def error(obj, text):
        return obj.BOLD + obj.ERROR + text + obj.ENDC + obj.ENDB


    ## ordinary text for logging ##

    @classmethod
    def title(obj, text):
        return obj.BOLD + obj.TITLE + text + obj.ENDC + obj.ENDB

    @classmethod
    def info(obj, text):
        return obj.INFO + text + obj.ENDC

    @classmethod
    def debug(obj, text):
        return obj.DEBUG + text + obj.ENDC


    ## ordinary text formatting ##

    @classmethod
    def highlight(obj, text):
        return obj.BOLD + text + obj.ENDC


    ## meta functions ##

    @classmethod
    def uncolor(obj, text):
        return re.sub(obj.UNCOL, '', text)

    @classmethod
    def unescape(obj, text):
        for v, k in obj.ESCAPE[::-1]:
            text = text.replace(k, v)
        return text

    @classmethod
    def escape(obj, text):
        for k, v in obj.ESCAPE:
            text = text.replace(k, v)
        return text

class ColorsTerminal(Colors):
    """Terminal colors."""
    OK      = "\033[38;5;2m"    # green
    WARN    = "\033[33m"        # yellow
    ERROR   = "\033[31m"        # red
    TITLE   = ""                # white
    INFO    = "\033[38;5;5m"    # magenta
    DEBUG   = "\033[1;36m"      # light blue
    BOLD    = "\033[1m"         # bold
    ENDC    = "\033[0m"         # end color
    ENDB    = "\033[0m"         # end bold
    UNCOL   = '\\033\[.*?m'
    ESCAPE  = []

class ColorsMarkup(Colors):
    """Console colors.
    """
    OK      = "[color=#00FF00]" # green
    WARN    = "[color=#FFFF00]" # yellow
    ERROR   = "[color=#FF0000]" # red
    TITLE   = "[color=#FF0000]" # red
    INFO    = "[color=#A52292]" # magenta
    DEBUG   = "[color=#4CCBE4]" # light blue
    BOLD    = "[b]"
    ENDC    = "[/color]"
    ENDB    = "[/b]"
    UNCOL   = '\[/?(color|b)(=.*?)?\]'
    #UNCOL   = '\[/?(color|b)(=#[\dA-Fa-f]+)?\]' # only permits hex color values
    ESCAPE  = [
            ('&', '&amp;'),
            ('[', '&bl;' ),
            (']', '&br;' ),
            ]

    # markup removal
    # advanced: search for proper tags, i.e. start and ending pairs
    # needs to be repeated until no more text is replaced
    # FIXME: doesn't work for "[size=<size>]..[/size] style tags
    #rx3 = re.compile('\[([^\]]+)\](.*?)\[/\\1\]') # replace with '\\2'
    # alt (to be checked) for "[size=<size>]..[/size] style tags
    #rx3 = re.compile('\[([^\]]+)(=[^\]+)?\](.*?)\[/\\1\]') # replace with '\\3'
    # simple: remove everything that looks like a tag
    #rx3 = re.compile('\[[^\]]+\]') # replace with ''

## EOF ##
