"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
import logging

# tagit imports
from tagit import config

# exports
__all__ = ('load_logging', )


## code ##

def load_logging(cfg):
    """Configure the main terminal logger."""
    if ('logging', ) in cfg:
        from tagit import loghandler
        from tagit.shared import ColorsTerminal
        from .logger import logger_config
        logger_config(loghandler, ColorsTerminal, cfg('logging').to_tree(defaults=True))

    if not cfg('session', 'debug'):
        # set to info, prevents any DEBUG message to pass through the system.
        # the handler config cannot overwrite this.
        logging.getLogger('tagit').setLevel(logging.INFO)


## config ##

config.declare_title(('logging', ), __name__, 'Logging',
    'Logging is used in several places (terminal, GUI). Wherever used, logging can be configured through a dictionary. Also see view.status, view.console for default values.')

config.declare(('logging', 'level'),
    config.Enum('debug', 'info', 'warn', 'error', 'critical'), 'debug',
    __name__, 'Log level', 'Maximal log level for which messages are shown. The order is: critical > error > warning > info > volatile > debug')

config.declare(('logging', 'filter'), config.List(config.String()), ['tagit'],
    __name__, 'Module filter', 'Module name for which log messages are accepted.')

config.declare(('logging', 'fmt'), config.String(), '[{levelname}] [{name}] {title}{message}',
    __name__, 'Log format', 'Log message formatting. The formatting follows the typical python format string where each item is enclosed in curly braces (e.g. "{message}"). For printable items see the `logging attributes <https://docs.python.org/3.6/library/logging.html#logrecord-attributes>`_.  In addition to those, the item "title" is available, which is a placeholder for the formatted title (if present).')

config.declare(('logging', 'title'), config.String(), '[{title}] ',
    __name__, 'Title format', 'Title formatting.')

config.declare(('logging', 'maxlen'), config.Unsigned(), 80,
    __name__, 'Maximal line length', 'Maximal line length (e.g. console width). Use Infinity to set no line length limit.')

config.declare(('logging', 'prefix'), config.String(), '',
    __name__, 'Log prefix', 'A prefix before every log line (internal use only)')

## EOF ##
