"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# external imports
from pyparsing import CaselessKeyword, Group, Or, Word, delimitedList, oneOf, ParseException

# tagit imports
from tagit.utils import errors, Struct

# exports
__all__ = (
    'Sort',
    )


## code ##

class Sort():
    """Sort parser.

    A sort string can be as simple as a predicate, but also allows
    a more verbose specification for more natural readability.
    In brief and somewhat relaxed notation, the syntax is:
        [sort [<type>] by] <predicate> [similarity to] [<anchor>] [<direction>]
    Multiple sort terms are concatenated with a comma.

    Examples:
        time
        time asc
        sort by time desc
        sort numerically by time downwards
        sort by tag similarity to AF39D281CE3 up
        time, iso
    """
    QUERY = None
    PREDICATES = None

    def __init__(self, sortkeys):
        self.sortkeys = sortkeys

    def __call__(self, query):
        return self.parse(query)

    def build_parser(self, predicate=None):
        # The *predicate* argument is for compatibility with predicate listener.
        # It's not actually used here.
        """
        The grammar is composed as follows:

        QUERY := EXPR | EXPR, EXPR
        EXPR := PREFIX PREDICATE SUFFIX | PREDICATE SUFFIX | PREFIX PREDICATE | PREDICATE
        PREFIX := sort TYPE by | sort by
        SUFFIX := SIMILAR DIRECTION | SIMILAR | DIRECTION
        SIMILAR := similarity to ANCHOR | ANCHOR
        TYPE := numerically | alphabetically
        PREDICATE := [predicate]
        ANCHOR := [guid]
        DIRECTION := up | down | asc | desc | ascending | descending | reversed | upwards | downwards
        """
        # predicates from sortkeys
        self.PREDICATES = self.sortkeys.scope.library | self.sortkeys.typedef.anchored

        ## terminals
        # direction is just a list of keywords
        direction = oneOf('up down asc desc ascending descending reversed upwards downwards',
                        caseless=True).setResultsName('direction')
        # type is just a list of keywords
        type_ = oneOf('numerically alphabetically').setResultsName('type')
        # predicates are from an enum
        predicate = Or([CaselessKeyword(p) for p in self.PREDICATES]).setResultsName('predicate')
        # anchor is a hex digest
        anchor = Word('abcdef0123456789ABCDEF').setResultsName('anchor')

        ## rules
        similar = Or([CaselessKeyword('similarity to') + anchor,
                      anchor])
        suffix = Or([similar + direction, similar,
                     direction])
        prefix = Or([CaselessKeyword('sort') + type_ + CaselessKeyword('by'),
                     CaselessKeyword('sort by')])
        expr = Group(Or([prefix + predicate + suffix,
                    predicate + suffix,
                    prefix + predicate,
                    predicate]))

        self.QUERY = delimitedList(expr, delim=',')
        return self

    def __del__(self):
        if self.QUERY is not None: # remove listener
            try:
                self.sortkeys.ignore(self.build_parser)
            except ImportError:
                # The import fails if python is shutting down.
                # In that case, the ignore becomes unnecessary anyway.
                pass

    def parse(self, sort):
        if self.QUERY is None:
            # initialize parser
            self.build_parser()
            # attach listener to receive future updates
            self.sortkeys.listen(self.build_parser)

        try:
            parsed = self.QUERY.parseString(sort, parseAll=True)
        except ParseException as e:
            raise errors.ParserError('Cannot parse query', e)

        # convert to AST
        tokens = []
        for exp in parsed:
            args = Struct(
                predicate=None,
                type=None,
                anchor=None,
                direction='asc',
                )
            args.update(**exp.asDict())

            # check predicate
            if args.predicate is None: # prevented by grammar
                raise errors.ParserError('Missing sort key', exp)
            if args.predicate not in self.sortkeys: # prevented by grammar
                raise errors.ParserError('Invalid sort key', exp)

            # check direction
            if args.direction in ('up', 'ascending', 'asc', 'upwards'):
                reverse = False
            elif args.direction in ('down', 'desc', 'descending', 'reversed', 'downwards'):
                reverse = True
            else: # prevented by grammar
                raise errors.ParserError('Invalid direction', exp)

            # infer type from predicate if needed
            if args.anchor is not None:
                args.type = 'anchored'
            elif args.type is None:
                typedef = self.sortkeys.predicate(args.predicate).typedef
                if not len(typedef):
                    raise errors.ParserError('Undefined type', exp)
                elif len(typedef) == 1:
                    args.type = list(typedef)[0].lower()
                else:
                    raise errors.ParserError('Ambiguous type', exp)

            # translate types
            args.type = {
                    'numerically': 'numerical',
                    'alphabetically': 'alphabetical'
                    }.get(args.type, args.type)

            # check type compatibility
            admissible_types = {t.lower() for t in self.sortkeys.predicate(args.predicate).typedef}
            if args.type not in admissible_types:
                raise errors.ParserError('Invalid type for predicate', exp)
            elif args.type == 'anchored' and args.anchor is None: # type set if anchor isn't None
                raise errors.ParserError('No anchor given', exp)

            # build AST
            if args.type in ('anchored', ):
                tokens.append(ast.AnchoredSort(args.predicate, args.anchor, reverse))
            elif args.type in ('alphabetical', 'alphabetically'):
                tokens.append(ast.AlphabeticalSort(args.predicate, reverse))
            elif args.type in ('numerical', 'numerically'):
                tokens.append(ast.NumericalSort(args.predicate, reverse))
            else: # prevented by grammar
                raise errors.ParserError('Invalid type for predicate', exp)

        # aggregate if need be
        if len(tokens) == 1:
            return tokens[0]
        else:
            return ast.Order(*tokens)

## EOF ##
