"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
from functools import reduce
import operator

# kivy imports
from kivy.lang import Builder

# tagit imports
from tagit.widgets.browser import BrowserAwareMixin

# inner-module imports
from .tile import TileWithLabel

# exports
__all__ = ('BrowserTags', )


## code ##

# load kv
Builder.load_string('''
<BrowserTags>:
    title: 'Tags'
    tooltip: 'Tags of displayed items.'
''')

# classes
class BrowserTags(TileWithLabel, BrowserAwareMixin):
    """Show tags of displayed items. Tags of selected items are highlighted."""

    displayed = None

    def on_browser(self, sender, browser):
        # remove old binding
        if self.browser is not None:
            self.browser.unbind(cursor=self.update)
            self.browser.unbind(selection=self.update)
            self.browser.unbind(items=self.on_items)
        # add new binding
        self.browser = browser
        if self.browser is not None:
            self.browser.bind(cursor=self.update)
            self.browser.bind(selection=self.update)
            self.browser.bind(items=self.on_items)
            # populate displayed first, then update
            self.on_items(browser, browser.items)
            self.update()

    def __del__(self):
        if self.browser is not None:
            self.browser.unbind(cursor=self.update)
            self.browser.unbind(selection=self.update)
            self.browser.unbind(items=self.on_items)
            self.browser = None

    def on_items(self, browser, items):
        # unfold
        items = browser.unfold(items)
        # get tags
        self.displayed = items.tag.label(node=False)
        # update view
        self.update()

    def update(self, *args):
        if not self.visible:
            self.text = ''

        elif self.displayed is None:
            self.on_items(self.root.browser, self.root.browser.items)
            # calls update again with not-None self.displayed

        else:
            browser = self.root.browser

            # handle cursor
            if browser.cursor is None:
                cursor = set()
            else:
                cursor = browser.cursor.tag.label()

            # handle selection
            if len(browser.selection) == 0:
                selected = set()
            else:
                selection = reduce(operator.add, browser.selection)
                selected = selection.tag.label(node=False)

            # assemble tag list
            tags = []
            for tag in sorted(self.displayed | selected | cursor):
                pretty = tag
                if tag in cursor:
                    pretty = f'[b]{pretty}[/b]' # bold
                if tag in selected:
                    pretty = f'[color=#415bCD]{pretty}[/color]' # blue color

                tags.append(pretty)

            # Apply prefix and display
            self.text = ', '.join(tags)

## EOF ##
