"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
from collections import OrderedDict

# kivy imports
from kivy.lang import Builder

# tagit imports
from tagit.utils import ttime, ns, magnitude_fmt
from tagit.widgets.browser import BrowserAwareMixin
from tagit.widgets.session import StorageAwareMixin

# inner-module imports
from .tile import TileTabular

# exports
__all__ = ('Info', )


## code ##

# load kv
Builder.load_string('''
<Info>:
    title: 'Item info'
    tooltip: 'Key properties of the cursor item'
    # assuming 7 info items
    #default_size: None, 7*self.font_size + 6*5
    keywidth: min(75, self.width * 0.4)
''')


# classes
class Info(TileTabular, BrowserAwareMixin, StorageAwareMixin):
    """Show essential attributes about the cursor."""

    def on_root(self, wx, root):
        BrowserAwareMixin.on_root(self, wx, root)
        StorageAwareMixin.on_root(self, wx, root)

    def on_browser(self, sender, browser):
        # remove old binding
        if self.browser is not None:
            self.browser.unbind(cursor=self.update)
        # add new binding
        self.browser = browser
        if self.browser is not None:
            self.browser.bind(cursor=self.update)
            self.update()

    def __del__(self):
        if self.browser is not None:
            self.browser.unbind(cursor=self.update)
            self.browser = None

    def on_predicate_modified(self, *args):
        self.update()

    def update(self, *args):
        cursor = self.root.browser.cursor
        if not self.visible or cursor is None:
            # invisible or no cursor, nothing to show
            self.tabledata = OrderedDict({})

        else:
            preds = cursor.get(
                ns.bsfs.Node().t_created,
                ns.bse.filesize,
                ns.bse.filename,
                (ns.bse.tag, ns.bst.label),
                )
            self.tabledata = OrderedDict({
                'Date'      : ttime.from_timestamp_utc(
                    preds.get(ns.bsfs.Node().t_created, ttime.timestamp_min)).strftime('%d.%m.%y %H:%M'),
                'Filesize'  : magnitude_fmt(preds.get(ns.bse.filesize, 0)),
                'Filename'  : preds.get(ns.bse.filename, 'n/a'),
                'Tags'      : ', '.join(sorted(preds.get((ns.bse.tag, ns.bst.label), [' ']))),
                })

## EOF ##
