"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
import json

# exports
__all__ = ('Frame', )


## code ##

class Frame(dict):
    def __init__(self, cursor=None, selection=None, offset=0, **kwargs):
        super(Frame, self).__init__(**kwargs)
        selection = selection if selection is not None else []
        self['cursor'] = cursor
        self['selection'] = selection
        self['offset'] = offset

    @property
    def selection(self):
        return self['selection']

    @property
    def cursor(self):
        return self['cursor']

    @property
    def offset(self):
        return self['offset']

    def copy(self):
        return Frame(**super(Frame, self).copy())

    def serialize(self):
        return json.dumps({
            'cursor': self.cursor.guid if self.cursor is not None else 'None',
            'group': self.cursor.group if hasattr(self.cursor, 'group') else 'None',
            'selection': [img.guid for img in self.selection],
            'offset': self.offset
        })

    @staticmethod
    def from_serialized(lib, serialized, ignore_errors=True):
        d = json.loads(serialized)

        # load cursor
        cursor = None
        try:
            if 'cursor' in d and d['cursor'] is not None and d['cursor'].lower() != 'none':
                cursor = lib.entity(d['cursor'])
        except KeyError as err:
            if not ignore_errors:
                raise err

        if 'group' in d and d['group'] is not None and d['group'].lower() != 'none':
            try:
                # FIXME: late import; breaks module dependency structure
                from tagit.storage.library.entity import Representative
                cursor = Representative.Representative(lib, d['group'])
            except ValueError:
                # group doesn't exist anymore; ignore
                pass

        # load selection
        selection = []
        for guid in d.get('selection', []):
            try:
                selection.append(lib.entity(guid))
            except KeyError as err:
                if not ignore_errors:
                    raise err

        return Frame(
            cursor = cursor,
            selection = selection,
            offset = d.get('offset', 0)
            )

## EOF ##
