"""

Part of the tagit module.
A copy of the license is provided with the project.
Author: Matthias Baumgartner, 2022
"""
# standard imports
from functools import partial
import logging
import os

# kivy imports
from kivy.clock import Clock
from kivy.config import Config as KivyConfig
from kivy.lang import Builder
from kivy.uix.behaviors import ButtonBehavior
from kivy.uix.behaviors import FocusBehavior
from kivy.uix.boxlayout import BoxLayout
from kivy.uix.image import Image
from kivy.uix.textinput import TextInput
import kivy.properties as kp

# tagit imports
from tagit import config
from tagit.utils import bsfs, errors, ns
from tagit.utils.bsfs import ast, matcher

# inner-module imports
from .session import ConfigAwareMixin

# exports
__all__ = ('Filter', )


## code ##

logger = logging.getLogger(__name__)

# load kv
Builder.load_file(os.path.join(os.path.dirname(__file__), 'filter.kv'))

# classes
class Filter(BoxLayout, ConfigAwareMixin):
    """
    A filter tracks a sequence of searches building on top of each other. Each
    item in that sequence is defined by a part of the overall search query
    (token). In addition, the filter also tracks the viewport at each point in
    the sequence (frames).

    In addition, the sequence can be navigated back-and-forth, so that the
    current search includes a number of items, starting at the front, but not
    necessarily all. Hence, some tokens are present in the current
    search (head), while others are not (tail).
    """
    # root reference
    root = kp.ObjectProperty(None)

    # change notification
    changed = kp.BooleanProperty(False)
    run_search = kp.BooleanProperty(False)

    # appearance
    MODE_SHINGLES = 'shingles'
    MODE_ADDRESS = 'address'
    searchmode = kp.OptionProperty(MODE_SHINGLES, options=[MODE_SHINGLES, MODE_ADDRESS])

    '''
    To track head, tail, tokens, and frames, four properties are used for
    the relevant pairwise combinations.

    For heads, the frame is the last known viewport before applying the
    next filter token. I.e. f_head[1] corresponds to the search including
    tokens t_head[:1]. The viewport of the current search is maintained
    in the browser.

    For tails, the frame is the last viewport before switching to the previous
    filter token. I.e. f_tail[1] corresponds to the search including
    tokens t_tail[:2] (i.e. the lists are aligned).

    Consider the following scheme.
    The current search is indicated by the "v". The first search includes
    no tokens (all items). Note the offset between tokens and frames in
    the head part.

                  v
    view  0 1 2 3 4
    token - 0 1 2 3 0 1
    frame 0 1 2 3 - 0 1

    Although the lists are not necessarily aligned, they always have to have
    the same size. This constraint is enforced.

    '''
    # tokens
    t_head = kp.ListProperty()
    t_tail = kp.ListProperty()

    # frames
    f_head = kp.ListProperty()
    f_tail = kp.ListProperty()

    # sort
    #sortkey = kp.ObjectProperty(partial(ast.NumericalSort, 'time'))
    sortkey = kp.ObjectProperty(None) # FIXME: mb/port
    sortdir = kp.BooleanProperty(False) # False means ascending


    ## exposed methods

    def get_query(self):
        query = bsfs.ast.filter.And(self.t_head[:]) if len(self.t_head) > 0 else None
        sort = None
        return query, sort
        # FIXME: mb/port.parsing
        query = ast.AND(self.t_head[:]) if len(self.t_head) else None
        # sort order is always set to False so that changing the sort order
        # won't trigger a new query which can be very expensive. The sort
        # order is instead applied in uix.kivy.actions.search.Search.
        sort  = self.sortkey(False) if self.sortkey is not None else None
        return query, sort

    def abbreviate(self, token):
        # FIXME: Return image
        matches = matcher.Filter()
        if matches(token, ast.filter.Any(ns.bse.tag, ast.filter.Any(ns.bst.label, matcher.Any()))):
            # tag token
            return 'T'
        if matches(token, matcher.Partial(ast.filter.Is)) or \
           matches(token, ast.filter.Or(matcher.Rest(matcher.Partial(ast.filter.Is)))):
            # exclusive token
            return '='
        if matches(token, ast.filter.Not(matcher.Partial(ast.filter.Is))) or \
           matches(token, ast.filter.Not(ast.filter.Or(matcher.Rest(matcher.Partial(ast.filter.Is))))):
            # reduce token
            return '—'
        if matches(token, ast.filter.Any(ns.bse.group, matcher.Any())):
            # group token
            return 'G'
        if matches(token, ast.filter.Any(matcher.Partial(ast.filter.Predicate), matcher.Any())):
            # generic token
            #return token.predicate.predicate.get('fragment', '?').title()[0]
            return 'P'
        return '?'

    def tok_label(self, token):
        matches = matcher.Filter()
        if matches(token, ast.filter.Any(ns.bse.tag, ast.filter.Any(ns.bst.label, matcher.Any()))):
            # tag token
            return self.root.session.filter_to_string(token)
        if matches(token, matcher.Partial(ast.filter.Is)) or \
           matches(token, ast.filter.Not(matcher.Partial(ast.filter.Is))):
            return '1'
        if matches(token, ast.filter.Or(matcher.Rest(matcher.Partial(ast.filter.Is)))):
            return str(len(token))
        if matches(token, ast.filter.Not(ast.filter.Or(matcher.Rest(matcher.Partial(ast.filter.Is))))):
            return str(len(token.expr))
        if matches(token, ast.filter.Any(matcher.Partial(ast.filter.Predicate), matcher.Any())):
            # generic token
            #return self.root.session.filter_to_string(token)
            return token.predicate.predicate.get('fragment', '')
        return ''

    def show_address_once(self):
        """Single-shot address mode without changing the search mode."""
        self.tokens.clear_widgets()
        searchbar = Addressbar(self.t_head, root=self.root)
        self.tokens.add_widget(searchbar)
        searchbar.focus = True


    ## initialization

    def on_config_changed(self, session, key, value):
        if key == ('ui', 'standalone', 'filter', 'searchbar'):
            self.on_cfg(session, session.cfg)

    def on_cfg(self, wx, cfg):
        with self:
            self.searchmode = cfg('ui', 'standalone', 'filter', 'searchbar')

    ## filter as context

    def __enter__(self):
        return self

    def  __exit__(self, exc_type, exc_value, traceback):
        if not(len(self.t_head) == len(self.f_head)):
            raise errors.ProgrammingError('head sizes differ')
        if not(len(self.t_tail) == len(self.f_tail)):
            raise errors.ProgrammingError('tail sizes differ')

        # issue redraw
        if self.changed:
            self.redraw()
        # issue search
        if self.run_search:
            self.root.trigger('Search')

    def redraw(self):
        self.tokens.clear_widgets()
        if self.searchmode == self.MODE_ADDRESS:
            # add address bar
            self.tokens.add_widget(Addressbar(self.t_head, root=self.root))

        elif self.searchmode == self.MODE_SHINGLES:
            # add shingles
            for tok in self.t_head + self.t_tail:
                self.tokens.add_widget(
                    Shingle(
                        tok,
                        active=(tok in self.t_head),
                        avatar=self.abbreviate(tok),
                        text=self.tok_label(tok),
                        root=self.root
                    ))

    ## property access

    def on_t_head(self, sender, t_head):
        self.changed = True
        self.run_search = True

    def on_t_tail(self, sender, t_tail):
        self.changed = True

    def on_searchmode(self, sender, mode):
        self.changed = True

    def on_sortdir(self, sender, sortdir):
        self.run_search = True

    def on_sortkey(self, sender, sortkey):
        self.run_search = True


class FilterAwareMixin(object):
    """Tile that binds to the filter."""
    filter = None
    def on_root(self, wx, root):
        root.bind(filter=self.on_filter)
        if root.filter is not None:
            # initialize with the current filter
            # Going through the event dispatcher ensures that the object
            # is initialized properly before on_filter is called.
            Clock.schedule_once(lambda dt: self.on_filter(root, root.filter))

    def on_filter(self, sender, filter):
        pass


class Shingle(BoxLayout):
    """A sequence of filter tokens. Tokens can be edited individually."""
    # root reference
    root = kp.ObjectProperty(None)

    # content
    active = kp.BooleanProperty(False)
    text = kp.StringProperty('')
    avatar = kp.StringProperty('')

    # touch behaviour
    _single_tap_action = None

    def __init__(self, token, **kwargs):
        super(Shingle, self).__init__(**kwargs)
        self.token = token

    def remove(self, *args, **kwargs):
        """Remove shingle."""
        self.root.trigger('RemoveToken', self.token)

    def on_touch_down(self, touch):
        """Edit shingle when touched."""
        if self.collide_point(*touch.pos):
            if touch.is_double_tap: # edit filter
                # ignore touch, such that the dialogue
                # doesn't loose the focus immediately after open
                if self._single_tap_action is not None:
                    self._single_tap_action.cancel()
                    self._single_tap_action = None
                FocusBehavior.ignored_touch.append(touch)
                self.root.trigger('EditToken', self.token)
                return True
            else: # jump to filter
                # delay executing the action until we're sure it's not a double tap
                self._single_tap_action = Clock.schedule_once(
                    lambda dt: self.root.trigger('JumpToToken', self.token),
                    KivyConfig.getint('postproc', 'double_tap_time') / 1000)
                return True

        return super(Shingle, self).on_touch_down(touch)

class Addressbar(TextInput):
    """An address bar where a search query can be entered and edited.
    Edits are accepted by pressing Enter and rejected by pressing Esc.
    """
    # root reference
    root = kp.ObjectProperty()

    def __init__(self, tokens, **kwargs):
        super(Addressbar, self).__init__(**kwargs)
        self.text = self.root.session.filter_to_string(bsfs.ast.filter.And(tokens))
        self._last_text = self.text

    def on_text_validate(self):
        """Accept text as search string."""
        self.root.trigger('SetToken', self.text)
        self._last_text = self.text

    def on_keyboard(self, *args, **kwargs):
        """Block key propagation to other widgets."""
        return True

    def on_focus(self, wx, focus):
        from kivy.core.window import Window
        if focus:
            # fetch keyboard
            Window.bind(on_keyboard=self.on_keyboard)
            # keep a copy of the current text
            self._last_text = self.text
        else:
            # release keyboard
            Window.unbind(on_keyboard=self.on_keyboard)
            # set last accepted text
            self.text = self._last_text


## config ##

config.declare(('ui', 'standalone', 'filter', 'searchbar'),
        config.Enum('shingles', 'address'), 'shingles',
        __name__, 'Searchbar mode', 'Show either list of shingles, one per search token, or a freely editable address bar.')

## EOF ##
